/*
 * Decompiled with CFR 0.152.
 */
package picard.arrays;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.util.CloseableIterator;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.Genotype;
import htsjdk.variant.variantcontext.GenotypeBuilder;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextBuilder;
import htsjdk.variant.variantcontext.VariantContextUtils;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterBuilder;
import htsjdk.variant.vcf.VCFEncoder;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFFormatHeaderLine;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFHeaderLine;
import htsjdk.variant.vcf.VCFHeaderLineType;
import htsjdk.variant.vcf.VCFInfoHeaderLine;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.arrays.ZCallPedFile;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.GenotypingArraysProgramGroup;

@CommandLineProgramProperties(summary="MergePedIntoVcf takes a single-sample ped file output from zCall and merges into a single-sample vcf file using several supporting files.A VCF, aka Variant Calling Format, is a text file for storing how a sequenced sample differs from the reference genome. <a href='https://samtools.github.io/hts-specs/VCFv4.2.pdf'></a>A PED file is a whitespace-separated text file for storing genotype information. <a href='http://zzz.bwh.harvard.edu/plink/data.shtml#ped'></a>A MAP file is a whitespace-separated text file for storing information about genetic distance. <a href='http://zzz.bwh.harvard.edu/plink/data.shtml#map'></a>A zCall thresholds file is a whitespace-separated text file for storing the thresholds for genotype clustering for a SNP as determined by zCall.<a href='https://www.ncbi.nlm.nih.gov/pmc/articles/PMC3463112/#SEC2title'></a><h4>Usage example:</h4><pre>java -jar picard.jar MergePedIntoVcf \\<br />      VCF=input.vcf \\<br />      PED=zcall.output.ped \\<br />      MAP=zcall.output.map \\<br />      ZCALL_T_FILE=zcall.thresholds.7.txt \\<br />      OUTPUT=output.vcf <br /></pre>", oneLineSummary="Program to merge a single-sample ped file from zCall into a single-sample VCF.", programGroup=GenotypingArraysProgramGroup.class)
@DocumentedFeature
public class MergePedIntoVcf
extends CommandLineProgram {
    static final String USAGE_DETAILS = "MergePedIntoVcf takes a single-sample ped file output from zCall and merges into a single-sample vcf file using several supporting files.A VCF, aka Variant Calling Format, is a text file for storing how a sequenced sample differs from the reference genome. <a href='https://samtools.github.io/hts-specs/VCFv4.2.pdf'></a>A PED file is a whitespace-separated text file for storing genotype information. <a href='http://zzz.bwh.harvard.edu/plink/data.shtml#ped'></a>A MAP file is a whitespace-separated text file for storing information about genetic distance. <a href='http://zzz.bwh.harvard.edu/plink/data.shtml#map'></a>A zCall thresholds file is a whitespace-separated text file for storing the thresholds for genotype clustering for a SNP as determined by zCall.<a href='https://www.ncbi.nlm.nih.gov/pmc/articles/PMC3463112/#SEC2title'></a><h4>Usage example:</h4><pre>java -jar picard.jar MergePedIntoVcf \\<br />      VCF=input.vcf \\<br />      PED=zcall.output.ped \\<br />      MAP=zcall.output.map \\<br />      ZCALL_T_FILE=zcall.thresholds.7.txt \\<br />      OUTPUT=output.vcf <br /></pre>";
    private final Log log = Log.getInstance(MergePedIntoVcf.class);
    private final ProgressLogger logger = new ProgressLogger(this.log, 10000);
    @Argument(shortName="VCF", doc="The vcf containing the original autocall genotypes.")
    public File ORIGINAL_VCF;
    @Argument(shortName="PED", doc="PED file to be merged into VCF.")
    public File PED_FILE;
    @Argument(shortName="MAP", doc="MAP file for the PED file.")
    public File MAP_FILE;
    @Argument(shortName="ZCALL_T_FILE", doc="The zcall thresholds file.")
    public File ZCALL_THRESHOLDS_FILE = null;
    @Argument(shortName="ZCALL_VERSION", doc="The version of zcall used")
    public String ZCALL_VERSION;
    @Argument(shortName="O", doc="The output VCF file to write with merged genotype calls.")
    public File OUTPUT;
    private static HashMap<String, String[]> zCallThresholds = new HashMap();

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable(this.PED_FILE);
        IOUtil.assertFileIsReadable(this.MAP_FILE);
        IOUtil.assertFileIsReadable(this.ZCALL_THRESHOLDS_FILE);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        try {
            this.parseZCallThresholds();
            ZCallPedFile zCallPedFile = ZCallPedFile.fromFile(this.PED_FILE, this.MAP_FILE);
            VCFFileReader vcfFileReader = new VCFFileReader(this.ORIGINAL_VCF, false);
            VCFHeader header = vcfFileReader.getFileHeader();
            if (header.getGenotypeSamples().size() > 1) {
                throw new PicardException("MergePedIntoVCF only works with single-sample VCFs.");
            }
            this.addAdditionalHeaderFields(header);
            this.writeVcf((CloseableIterator<VariantContext>)vcfFileReader.iterator(), this.OUTPUT, vcfFileReader.getFileHeader().getSequenceDictionary(), header, zCallPedFile);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        return 0;
    }

    private void parseZCallThresholds() {
        String notApplicable = "NA";
        try (Stream<String> stream = Files.lines(this.ZCALL_THRESHOLDS_FILE.toPath());){
            stream.forEach(line -> {
                String[] tokens = line.split("\t");
                if (tokens[1].equals("NA") || tokens[2].equals("NA")) {
                    if (!tokens[1].equals("NA") || !tokens[2].equals("NA")) {
                        throw new PicardException("Thresholds should either both exist or both not exist.");
                    }
                    zCallThresholds.put(tokens[0], new String[]{".", "."});
                } else {
                    zCallThresholds.put(tokens[0], new String[]{tokens[1], tokens[2]});
                }
            });
        }
        catch (IOException e) {
            throw new PicardException("Error parsing ZCall Thresholds File", e);
        }
    }

    private void addAdditionalHeaderFields(VCFHeader header) {
        header.addMetaDataLine(new VCFHeaderLine("zcallVersion", this.ZCALL_VERSION));
        header.addMetaDataLine(new VCFHeaderLine("zcallThresholds", this.ZCALL_THRESHOLDS_FILE.getName()));
        header.addMetaDataLine(new VCFInfoHeaderLine("zthresh_X", 1, VCFHeaderLineType.Float, "zCall X threshold"));
        header.addMetaDataLine(new VCFInfoHeaderLine("zthresh_Y", 1, VCFHeaderLineType.Float, "zCall Y threshold"));
        header.addMetaDataLine(new VCFFormatHeaderLine("GTA", 1, VCFHeaderLineType.String, "Illumina Autocall Genotype"));
        header.addMetaDataLine(new VCFFormatHeaderLine("GTZ", 1, VCFHeaderLineType.String, "zCall Genotype"));
    }

    private void writeVcf(CloseableIterator<VariantContext> variants, File output, SAMSequenceDictionary dict, VCFHeader vcfHeader, ZCallPedFile zCallPedFile) {
        try (VariantContextWriter writer = new VariantContextWriterBuilder().setOutputFile(output).setReferenceDictionary(dict).setOptions(VariantContextWriterBuilder.DEFAULT_OPTIONS).build();){
            writer.writeHeader(vcfHeader);
            while (variants.hasNext()) {
                VariantContext context = (VariantContext)variants.next();
                VariantContextBuilder builder = new VariantContextBuilder(context);
                if (zCallThresholds.containsKey(context.getID())) {
                    String[] zThresh = zCallThresholds.get(context.getID());
                    builder.attribute("zthresh_X", zThresh[0]);
                    builder.attribute("zthresh_Y", zThresh[1]);
                }
                Genotype originalGenotype = context.getGenotype(0);
                Map<String, Object> newAttributes = originalGenotype.getExtendedAttributes();
                VCFEncoder vcfEncoder = new VCFEncoder(vcfHeader, false, false);
                Map<Allele, String> alleleMap = vcfEncoder.buildAlleleStrings(context);
                String zCallAlleles = zCallPedFile.getAlleles(context.getID());
                if (zCallAlleles == null) {
                    throw new PicardException("No zCall alleles found for snp " + context.getID());
                }
                List<Allele> zCallPedFileAlleles = this.buildNewAllelesFromZCall(zCallAlleles, context.getAttributes());
                newAttributes.put("GTA", alleleMap.get(originalGenotype.getAllele(0)) + "/" + alleleMap.get(originalGenotype.getAllele(1)));
                newAttributes.put("GTZ", alleleMap.get(zCallPedFileAlleles.get(0)) + "/" + alleleMap.get(zCallPedFileAlleles.get(1)));
                Genotype newGenotype = GenotypeBuilder.create(originalGenotype.getSampleName(), zCallPedFileAlleles, newAttributes);
                builder.genotypes(newGenotype);
                this.logger.record("0", 0);
                VariantContextUtils.calculateChromosomeCounts(builder, false);
                VariantContext newContext = builder.make();
                writer.add(newContext);
            }
        }
    }

    private List<Allele> buildNewAllelesFromZCall(String zCallPedFileAlleles, Map<String, Object> newAttributes) {
        char allele1 = zCallPedFileAlleles.charAt(0);
        char allele2 = zCallPedFileAlleles.charAt(1);
        ArrayList<Allele> newAlleles = new ArrayList<Allele>();
        String alleleA = String.valueOf(newAttributes.get("ALLELE_A"));
        String alleleB = String.valueOf(newAttributes.get("ALLELE_B"));
        newAlleles.add(this.translateAllele(alleleA, alleleB, allele1));
        newAlleles.add(this.translateAllele(alleleA, alleleB, allele2));
        return newAlleles;
    }

    private Allele translateAllele(String alleleA, String alleleB, char allele) {
        if (allele == 'A') {
            return this.formatAllele(alleleA);
        }
        if (allele == 'B') {
            return this.formatAllele(alleleB);
        }
        if (allele == '0') {
            return Allele.NO_CALL;
        }
        throw new PicardException("Illegal allele: " + allele);
    }

    private Allele formatAllele(String alleleA) {
        if (alleleA.equals("*")) {
            return Allele.SPAN_DEL;
        }
        if (alleleA.contains("*")) {
            return Allele.create(alleleA.replace("*".charAt(0), ' ').trim(), true);
        }
        return Allele.create(alleleA, false);
    }
}

